<?php

/**
 * Copyright 2014 Fabian Grutschus. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are those
 * of the authors and should not be interpreted as representing official policies,
 * either expressed or implied, of the copyright holders.
 *
 * @author    Fabian Grutschus <f.grutschus@lubyte.de>
 * @copyright 2014 Fabian Grutschus. All rights reserved.
 * @license   BSD
 * @link      http://github.com/fabiang/xmpp
 */

namespace Fabiang\Xmpp\EventListener\Stream;

use PHPUnit\Framework\TestCase;
use Fabiang\Xmpp\Event\XMLEvent;
use Fabiang\Xmpp\Connection\Test;
use Fabiang\Xmpp\Event\EventManager;
use Fabiang\Xmpp\Options;

/**
 * Generated by PHPUnit_SkeletonGenerator 1.2.1 on 2014-01-11 at 18:29:57.
 *
 * @coversDefaultClass Fabiang\Xmpp\EventListener\Stream\Authentication
 */
class AuthenticationTest extends TestCase
{

    /**
     * @var Authentication
     */
    protected $object;

    /**
     *
     * @var Test
     */
    protected $connection;

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     *
     * @return void
     */
    protected function setUp()
    {
        $this->object = new Authentication;
        $this->connection = new Test;

        $options = new Options;
        $options->setConnection($this->connection);
        $this->object->setOptions($options);
        $this->connection->setReady(true);
    }

    /**
     * Test what events are attached.
     *
     * @covers ::attachEvents
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Event\EventManager
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Stream\XMLStream
     * @return void
     */
    public function testAttachEvents()
    {
        $connection = new Test();
        $options = new Options;
        $options->setConnection($connection);
        $this->object->setOptions($options);
        $this->object->attachEvents();

        $input = $connection->getInputStream()->getEventManager();
        $this->assertArrayHasKey('{urn:ietf:params:xml:ns:xmpp-sasl}mechanisms', $input->getEventList());
        $this->assertArrayHasKey('{urn:ietf:params:xml:ns:xmpp-sasl}mechanism', $input->getEventList());
    }

    /**
     * Test collecting machanisms from event.
     *
     * @covers ::collectMechanisms
     * @covers ::getMechanisms
     * @covers ::isBlocking
     * @covers ::isAuthenticated
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Event\Event
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @uses Fabiang\Xmpp\Options
     * @return void
     */
    public function testCollectMechanisms()
    {
        $element = new \DOMElement('machanism', 'PLAIN');
        $event   = new XMLEvent;
        $event->setParameters(array($element));
        $this->object->collectMechanisms($event);
        $this->assertSame(array('plain'), $this->object->getMechanisms());

        $element = new \DOMElement('machanism', 'DIGEST-MD5');
        $event->setParameters(array($element));
        $this->object->collectMechanisms($event);
        $this->assertSame(array('plain', 'digest-md5'), $this->object->getMechanisms());

        $this->assertTrue($this->object->isBlocking());
    }

    /**
     * Test authentication.
     *
     * @covers ::authenticate
     * @covers ::determineMechanismClass
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::collectMechanisms
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::isAuthenticated
     * @uses Fabiang\Xmpp\Util\XML
     * @uses Fabiang\Xmpp\Event\Event
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Event\EventManager
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Stream\XMLStream
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication\Plain
     * @return void
     */
    public function testAuthenticate()
    {
        $this->object->setEventManager(new EventManager);
        $this->object->getOptions()->setUsername('aaa')
            ->setPassword('bbb');

        $element = new \DOMElement('machanism', 'PLAIN');
        $event   = new XMLEvent;
        $event->setParameters(array($element));
        $this->object->collectMechanisms($event);

        $element = new \DOMElement('mechanisms');
        $event   = new XMLEvent;
        $event->setParameters(array($element));

        $this->object->authenticate($event);
        $this->assertContains(
            '<auth xmlns="urn:ietf:params:xml:ns:xmpp-sasl" mechanism="PLAIN">AGFhYQBiYmI=</auth>',
            $this->connection->getBuffer()
        );
    }

    /**
     * Test authentication when no mechanisms where collected.
     *
     * @covers ::authenticate
     * @covers ::determineMechanismClass
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::isAuthenticated
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Event\Event
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @expectedException \Fabiang\Xmpp\Exception\RuntimeException
     * @expectedExceptionMessage No supportet authentication machanism found.
     * @return void
     */
    public function testAuthenticateWithoutMachanism()
    {
        $element = new \DOMElement('mechanisms');
        $event   = new XMLEvent;
        $event->setParameters(array($element));

        $this->object->authenticate($event);
    }

    /**
     * Test authentication when mechanism class is invalid instance.
     *
     * @covers ::authenticate
     * @covers ::determineMechanismClass
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::collectMechanisms
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::isAuthenticated
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Event\Event
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @expectedException \Fabiang\Xmpp\Exception\RuntimeException
     * @return void
     */
    public function testAuthenticateInvalidMechanismHandler()
    {
        $this->object->getOptions()->setAuthenticationClasses(array('plain' => '\stdClass'));

        $element = new \DOMElement('machanism', 'PLAIN');
        $event   = new XMLEvent;
        $event->setParameters(array($element));
        $this->object->collectMechanisms($event);

        $element = new \DOMElement('mechanisms');
        $event   = new XMLEvent;
        $event->setParameters(array($element));

        $this->object->authenticate($event);
    }

    /**
     * Test authentication failure.
     *
     * @covers ::failure
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Event\EventManager
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Stream\XMLStream
     * @uses Fabiang\Xmpp\Event\Event
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @uses Fabiang\Xmpp\Exception\Stream\StreamErrorException
     * @expectedException Fabiang\Xmpp\Exception\Stream\StreamErrorException
     * @return void
     */
    public function testFailure()
    {
        $document = new \DOMDocument;
        $element = new \DOMElement('failure');
        $document->appendChild($element);
        $event   = new XMLEvent;
        $event->setParameters(array($element));

        try {
            $this->object->failure($event);
        } catch (\Fabiang\Xmpp\Exception\StreamErrorException $e) {
            $this->assertFalse($this->object->isBlocking());
            $this->assertSame('<failure/>', $e->getContent());
            throw $e;
        }
    }

    /**
     * Test successful authentication.
     *
     * @covers ::success
     * @uses Fabiang\Xmpp\EventListener\Stream\Authentication::isBlocking
     * @uses Fabiang\Xmpp\EventListener\AbstractEventListener
     * @uses Fabiang\Xmpp\Connection\AbstractConnection
     * @uses Fabiang\Xmpp\Options
     * @uses Fabiang\Xmpp\Event\XMLEvent
     * @uses Fabiang\Xmpp\Event\Event
     * @return void
     */
    public function testSuccess()
    {
        $element = new \DOMElement('success');
        $event   = new XMLEvent;
        $event->setParameters(array($element));

        $connection = $this->createMock(Test::class);
        $this->object->getOptions()->setConnection($connection);

        $connection->expects($this->once())
            ->method('resetStreams');
        $connection->expects($this->once())
            ->method('connect');

        $this->object->success($event);
        $this->assertFalse($this->object->isBlocking());
        $this->assertTrue($this->object->getOptions()->isAuthenticated());
    }

}
